<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Pricing Helper for Wholesale and Retail Pricing
 * 
 * This helper provides functions to calculate wholesale and retail prices
 * based on purchase price and margin settings
 */

if (!function_exists('calculate_wholesale_price')) {
    /**
     * Calculate wholesale price based on purchase price and margin
     * 
     * @param float $purchase_price The purchase price of the item
     * @param string $margin_type 'percentage' or 'fixed'
     * @param float $margin_value The margin value
     * @return float The calculated wholesale price
     */
    function calculate_wholesale_price($purchase_price, $margin_type = 'percentage', $margin_value = 0) {
        if ($margin_type == 'percentage') {
            return $purchase_price + ($purchase_price * $margin_value / 100);
        } else {
            return $purchase_price + $margin_value;
        }
    }
}

if (!function_exists('calculate_retail_price')) {
    /**
     * Calculate retail price based on purchase price and margin
     * 
     * @param float $purchase_price The purchase price of the item
     * @param string $margin_type 'percentage' or 'fixed'
     * @param float $margin_value The margin value
     * @return float The calculated retail price
     */
    function calculate_retail_price($purchase_price, $margin_type = 'percentage', $margin_value = 0) {
        if ($margin_type == 'percentage') {
            return $purchase_price + ($purchase_price * $margin_value / 100);
        } else {
            return $purchase_price + $margin_value;
        }
    }
}

if (!function_exists('get_customer_price_type')) {
    /**
     * Get the price type for a customer (wholesale or retail)
     * 
     * @param int $customer_id The customer ID
     * @return string 'wholesale' or 'retail'
     */
    function get_customer_price_type($customer_id) {
        $CI =& get_instance();
        $CI->load->model('customers_model');
        
        $customer = $CI->customers_model->get_details($customer_id, array());
        return isset($customer['default_price_type']) ? $customer['default_price_type'] : 'retail';
    }
}

if (!function_exists('get_store_default_price_type')) {
    /**
     * Get the store's default price type
     * 
     * @return string 'wholesale' or 'retail'
     */
    function get_store_default_price_type() {
        $CI =& get_instance();
        $CI->load->model('store_model');
        
        $store_id = get_current_store_id();
        $store = $CI->store_model->get_details($store_id, array());
        return isset($store['default_price_type']) ? $store['default_price_type'] : 'retail';
    }
}

if (!function_exists('get_item_price_by_type')) {
    /**
     * Get item price based on price type
     * 
     * @param int $item_id The item ID
     * @param string $price_type 'wholesale' or 'retail'
     * @return float The price for the specified type
     */
    function get_item_price_by_type($item_id, $price_type = 'retail') {
        $CI =& get_instance();
        $CI->load->model('items_model');
        
        $item = $CI->items_model->get_details($item_id, array());
        
        if ($price_type == 'wholesale') {
            return isset($item['wholesale_price']) ? $item['wholesale_price'] : $item['sales_price'];
        } else {
            return isset($item['retail_price']) ? $item['retail_price'] : $item['sales_price'];
        }
    }
}

if (!function_exists('format_price_type_label')) {
    /**
     * Format price type as a label with styling
     * 
     * @param string $price_type 'wholesale' or 'retail'
     * @return string HTML formatted label
     */
    function format_price_type_label($price_type) {
        $label_class = ($price_type == 'wholesale') ? 'label-success' : 'label-info';
        $label_text = ucfirst($price_type);
        
        return '<span class="label ' . $label_class . '">' . $label_text . '</span>';
    }
}

if (!function_exists('auto_calculate_item_prices')) {
    /**
     * Auto calculate wholesale and retail prices for an item
     * 
     * @param int $item_id The item ID
     * @return bool Success status
     */
    function auto_calculate_item_prices($item_id) {
        $CI =& get_instance();
        $CI->load->model('items_model');
        
        $item = $CI->items_model->get_details($item_id, array());
        
        if (!$item || !$item['auto_calculate_prices']) {
            return false;
        }
        
        $purchase_price = $item['purchase_price'];
        $wholesale_price = calculate_wholesale_price(
            $purchase_price, 
            $item['wholesale_margin_type'], 
            $item['wholesale_margin_value']
        );
        $retail_price = calculate_retail_price(
            $purchase_price, 
            $item['retail_margin_type'], 
            $item['retail_margin_value']
        );
        
        // Update the item with calculated prices
        $CI->db->where('id', $item_id);
        return $CI->db->update('db_items', array(
            'wholesale_price' => $wholesale_price,
            'retail_price' => $retail_price
        ));
    }
}

/**
 * Get store default price type
 */
function get_store_default_price_type($store_id = null) {
    $CI =& get_instance();
    
    if (!$store_id) {
        $store_id = get_current_store_id();
    }
    
    $CI->db->select('default_price_type');
    $CI->db->where('id', $store_id);
    $query = $CI->db->get('db_store');
    
    if ($query->num_rows() > 0) {
        return $query->row()->default_price_type;
    }
    
    return 'retail'; // Default fallback
}

/**
 * Get item price based on price type
 */
function get_item_price_by_type($item_id, $price_type = null, $store_id = null) {
    $CI =& get_instance();
    
    if (!$price_type) {
        $price_type = get_store_default_price_type($store_id);
    }
    
    $CI->db->select('purchase_price, wholesale_price, retail_price, sales_price');
    $CI->db->where('id', $item_id);
    $query = $CI->db->get('db_items');
    
    if ($query->num_rows() > 0) {
        $item = $query->row();
        
        switch ($price_type) {
            case 'wholesale':
                return $item->wholesale_price;
            case 'retail':
                return $item->retail_price;
            case 'purchase':
                return $item->purchase_price;
            default:
                return $item->sales_price;
        }
    }
    
    return 0;
}
